<?php

namespace App\supplier1;

use App\Services\LogManager;

class CategoryComparison {
    private $logManager;
    private $inputDir;
    private $outputDir;
    private $categoryFile;
    private $foldersInitialized = false; // Flag to check if directories are set up

    public function __construct() {
        $this->logManager = new LogManager();
        $this->inputDir = __DIR__ . '/../../data/supplier1/';
        $this->categoryFile = $this->inputDir . 'Category-Website.csv';
    }

    // Move the folder setup here and ensure it's only done when needed
    private function setupDirectories() {
        if ($this->foldersInitialized) {
            return;  // Prevent repeated initialization
        }

        // Set up the output directory for category comparison
        $this->outputDir = __DIR__ . '/../../data-extracted/supplier1/category-comparison/category-comparison-' . date("Y-m-d-H-i-s") . '/';

        // Ensure the output directory exists, if not create it
        if (!is_dir($this->outputDir)) {
            mkdir($this->outputDir, 0777, true);
            $this->logManager->logInfo("Created output directory: " . $this->outputDir, 'categorycomparison', 'supplier1');
        }

        // Ensure input directory exists
        if (!is_dir($this->inputDir)) {
            $this->logManager->logError("Input directory does not exist: " . $this->inputDir, 'categorycomparison', 'supplier1');
            throw new \Exception("Input directory does not exist: " . $this->inputDir);
        }

        $this->foldersInitialized = true; // Mark as initialized
    }

    private function getLatestSecondaryCategoryFile() {
        $splitterDir = __DIR__ . '/../../data-extracted/supplier1/category-splitter/';
        $latestDir = null;
        $latestTime = 0;

        if (is_dir($splitterDir)) {
            $dirs = scandir($splitterDir);
            foreach ($dirs as $dir) {
                if ($dir === '.' || $dir === '..') continue;

                $dirPath = $splitterDir . $dir;
                if (is_dir($dirPath)) {
                    $dirTime = filemtime($dirPath);
                    if ($dirTime > $latestTime) {
                        $latestTime = $dirTime;
                        $latestDir = $dirPath;
                    }
                }
            }
        }

        if ($latestDir) {
            $latestFile = $latestDir . '/secondary-categories.csv';
            if (file_exists($latestFile)) {
                return $latestFile;
            } else {
                throw new \Exception("Secondary categories file does not exist in the latest directory: " . $latestFile);
            }
        } else {
            throw new \Exception("No directories found in category-splitter");
        }
    }

    public function compareCategories() {
        // Set up directories only when required
        $this->setupDirectories();

        if (!file_exists($this->categoryFile)) {
            $this->logManager->logError("Category file does not exist: " . $this->categoryFile, 'categorycomparison', 'supplier1');
            throw new \Exception("Category file does not exist: " . $this->categoryFile);
        }

        $catHeadArr = $this->createCatArray($this->categoryFile);

        // Get the latest secondary categories CSV file
        $guideFile = $this->getLatestSecondaryCategoryFile();

        if (file_exists($guideFile)) {
            $columnRequired = 1;
            $subkeysGuide = $this->csvColumnToArrayNoHeaders($guideFile, ',', $columnRequired);

            $subKeysAll = array_unique($subkeysGuide, SORT_REGULAR);
            $items = array_count_values($subKeysAll);

            $additions = [];
            $removals = [];

            foreach ($subKeysAll as $searchString) {
                $err = 0;
                foreach ($catHeadArr as $header => $value) {
                    $searchString = $this->hyperify($searchString);
                    $success = in_array($searchString, $value);
                    if ($success) {
                        $err = 1;
                    }
                }
                if ($err == 0) {
                    $additions[] = $searchString;
                }
            }

            $subKeysNew = [];
            foreach ($subKeysAll as $subValue) {
                $subKeysNew[] = $this->hyperify($subValue);
            }

            foreach ($catHeadArr as $key => $value) {
                foreach ($value as $searchString) {
                    $searchString = $this->hyperify($searchString);
                    $success = in_array($searchString, $subKeysNew);

                    if (!$success) {
                        $removals[] = $searchString;
                    }
                }
            }

            $this->saveResultsToCsv($additions, 'categories_to_add.csv');
            $this->saveResultsToCsv($removals, 'categories_to_remove.csv');

            $this->logManager->logInfo("Category comparison completed successfully", 'categorycomparison', 'supplier1');
            
            // Automatically process the category updates after generating the CSVs
            $this->processCategoryUpdates();  // Call the processing function here to run automatically

        } else {
            $this->logManager->logError("Files not found - please check", 'categorycomparison', 'supplier1');
            echo "Files not found - please check<br>";
            echo "Guide $guideFile<br>";
        }
    }

    private function saveResultsToCsv($data, $filename) {
        $filePath = $this->outputDir . $filename;
        $file = fopen($filePath, 'w');
        if (!$file) {
            $this->logManager->logError("Failed to open file for writing: $filePath", 'categorycomparison', 'supplier1');
            return;
        }
        foreach ($data as $value) {
            fputcsv($file, [$value]);
        }
        fclose($file);
        $this->logManager->logInfo("Saved results to CSV: $filePath", 'categorycomparison', 'supplier1');
    }

    // Non optimized one
    // private function createCatArray($csvFile = '', $delimiter = ',') {
    //     if (!file_exists($csvFile) || !is_readable($csvFile)) {
    //         exit("Your file doesn't exist or is not readable");
    //     }

    //     $header = null;
    //     $dataset = [];
    //     $delimiter = ",";

    //     if (($handle = fopen($csvFile, 'r')) !== false) {
    //         while (($row = fgetcsv($handle, 0, $delimiter)) !== false) {
    //             if (!$header) {
    //                 $header = $row;
    //             } else {
    //                 $x = 0;
    //                 foreach ($row as $cell) {
    //                     if ($cell) {
    //                         $cell = trim($cell);
    //                         $dataset[$header[$x]][] = $cell;
    //                     }
    //                     $x++;
    //                 }
    //             }
    //         }
    //         fclose($handle);
    //     }
    //     return $dataset;
    // }

    private function createCatArray($csvFile = '', $delimiter = ',') {
        if (!file_exists($csvFile) || !is_readable($csvFile)) {
            exit("Your file doesn't exist or is not readable");
        }
    
        $file = new \SplFileObject($csvFile);
        $file->setFlags(\SplFileObject::READ_CSV);
        $file->setCsvControl($delimiter);
    
        $header = null;
        $dataset = [];
        foreach ($file as $row) {
            if ($file->key() === 0) {
                $header = $row;
            } else {
                foreach ($row as $x => $cell) {
                    if ($cell) {
                        $dataset[$header[$x]][] = trim($cell);
                    }
                }
            }
        }
        return $dataset;
    }

    // NonOptimized one
    // private function csvColumnToArrayNoHeaders($file = '', $delimiter = ',', $column = '0') {
    //     if (!file_exists($file) || !is_readable($file)) {
    //         throw new \Exception($file . " doesn't exist or is not readable.");
    //     }

    //     $colArray = [];
    //     if (($handle = fopen($file, 'r')) !== false) {
    //         while (($data = fgetcsv($handle, 1000, $delimiter)) !== false) {
    //             $colArray[] = $data[$column];
    //         }
    //         fclose($handle);
    //     }
    //     return $colArray;
    // }

    private function csvColumnToArrayNoHeaders($file = '', $delimiter = ',', $column = '0') {
        if (!file_exists($file) || !is_readable($file)) {
            throw new \Exception($file . " doesn't exist or is not readable.");
        }
    
        $file = new \SplFileObject($file);
        $file->setFlags(\SplFileObject::READ_CSV);
        $file->setCsvControl($delimiter);
    
        $colArray = [];
        foreach ($file as $row) {
            $colArray[] = $row[$column] ?? null;
        }
        return array_filter($colArray);
    }

    private function hyperify($data) {
        $data = strtolower($data);

        if (preg_match("/-/", $data)) {
            // Hyphens
            $result = explode("-", $data);
            foreach ($result as $value) {
                $return[] = ucfirst(strtolower($value));
            }
            $return = implode("-", $return);
        } else {
            // All the other exceptions
            $data = ucwords($data);
            $data = str_replace("And", "and", $data);
            $data = str_replace("&", "and", $data);
            $data = str_replace("Ipad", "iPad", $data);
            $data = str_replace("Iphone", "iPhone", $data);
            $data = str_replace("Rfid", "RFID", $data);
            $data = str_replace("Usb", "USB", $data);
            $return = $data;
        }

        return $return;
    }

    public function processCategoryUpdates() {
        $categoryFile = $this->categoryFile; // The main category CSV
        $addFile = $this->outputDir . 'categories_to_add.csv';
        $removeFile = $this->outputDir . 'categories_to_remove.csv';
    
        // Load the main category CSV
        $categoryData = $this->createCatArray($categoryFile);
    
        // Process removals
        if (file_exists($removeFile)) {
            $this->removeCategories($categoryData, $removeFile);
        }
    
        // Process additions
        if (file_exists($addFile)) {
            $this->addCategories($categoryData, $addFile);
        }
    
        // Save the updated main CSV
        $this->saveMainCategoryCsv($categoryData, $categoryFile);
    }
    
    /**
     * Remove categories from the main CSV
     */
    private function removeCategories(&$categoryData, $removeFile) {
        $removals = $this->csvColumnToArrayNoHeaders($removeFile);
    
        // Remove quotes from the categories in the remove list
        $removals = array_map(function($value) {
            return trim(str_replace('"', '', $value));
        }, $removals);
    
        foreach ($removals as $subCategory) {
            foreach ($categoryData as $mainCategory => $subCategories) {
                $index = array_search($subCategory, $subCategories);
                if ($index !== false) {
                    // Remove the subcategory
                    unset($categoryData[$mainCategory][$index]);
                    $this->logManager->logInfo("Removed subcategory: $subCategory from $mainCategory", 'categorycomparison', 'supplier1');
    
                    // Reindex the array to remove any gaps
                    $categoryData[$mainCategory] = array_values($categoryData[$mainCategory]);
                }
            }
        }
    }
    
    /**
     * Add categories to the main CSV
     */
    private function addCategories(&$categoryData, $addFile) {
        // Load subcategories to add
        $additions = $this->csvColumnToArrayNoHeaders($addFile);
    
        // Start generating the HTML form
        echo "<form method='POST' action='index.php?action=process_categories' style='max-width: 800px; margin: auto;'>";
    
        // Begin the table
        echo "<table style='width: 100%; border-collapse: collapse; margin: 20px auto; text-align: center;'>";
        echo "<thead>";
        echo "<tr>";
        echo "<th style='border: 1px solid #ddd; padding: 10px; background-color: #f4f4f4;'>Subcategories</th>";
        
        // Add main category headers dynamically
        foreach (array_keys($categoryData) as $mainCategory) {
            echo "<th style='border: 1px solid #ddd; padding: 10px; background-color: #f4f4f4;'>$mainCategory</th>";
        }
        echo "</tr>";
        echo "</thead>";
        echo "<tbody>";
    
        // Populate rows with subcategories and checkboxes
        foreach ($additions as $subCategory) {
            echo "<tr>";
            echo "<td style='border: 1px solid #ddd; padding: 10px;'>$subCategory</td>";
    
            foreach (array_keys($categoryData) as $mainCategory) {
                echo "<td style='border: 1px solid #ddd; padding: 10px;'>
                        <input type='checkbox' name='main_category[$subCategory][]' value='$mainCategory'>
                      </td>";
            }
            echo "</tr>";
        }
    
        echo "</tbody>";
        echo "</table>";
    
        // Add the submit button
        echo "<input type='submit' value='Submit' style='display: block; width: 100%; padding: 10px; background-color: #4CAF50; color: #ffffff; border: none; border-radius: 5px; font-size: 1em; cursor: pointer;'>";
        echo "</form>";
    }
    
    /**
     * Save the updated main category CSV
     */
    private function saveMainCategoryCsv($categoryData, $filePath) {
        $file = fopen($filePath, 'w');
        if (!$file) {
            $this->logManager->logError("Failed to open file for writing: $filePath", 'categorycomparison', 'supplier1');
            return;
        }
    
        // Write main categories as the first row
        fputcsv($file, array_keys($categoryData));
    
        // Write subcategories under each main category
        $maxRows = max(array_map('count', $categoryData));
        for ($i = 0; $i < $maxRows; $i++) {
            $row = [];
            foreach ($categoryData as $subCategories) {
                $row[] = isset($subCategories[$i]) ? $subCategories[$i] : '';
            }
            fputcsv($file, $row);
        }
    
        fclose($file);
        $this->logManager->logInfo("Saved updated category file: $filePath", 'categorycomparison', 'supplier1');
    }

    public function assignCategories($selectedCategories) {
        $categoryFile = $this->categoryFile;  // Main category CSV file
        $categoryData = $this->createCatArray($categoryFile);  // Load the main CSV data
    
        foreach ($selectedCategories as $subcategory => $mainCategories) {
            // Check if subcategory and main categories are not empty
            if (!empty($subcategory) && !empty($mainCategories)) {
                foreach ($mainCategories as $mainCategory) {
                    // Add each subcategory to the appropriate main category
                    $categoryData[$mainCategory][] = $subcategory;
                    $this->logManager->logInfo("Added subcategory: $subcategory to $mainCategory", 'categorycomparison', 'supplier1');
                }
            }
        }
    
        // Save the updated category data to the main CSV file
        $this->saveMainCategoryCsv($categoryData, $categoryFile);
    }
    
}
